/*! \file                       DButton.h
    \brief                      Archivo que contiene las definiciones para la clase DButton, utilizada en botones de windows.

    \details                    El button es el control por excelencia de windows, y se puede utilizar con muchas finalidades. En esencia es un espacio donde hacemos click con el mouse para realizar alguna accin.\n\n
                                Ademas y pese a que no me acabe de gustar, el button no solo es el tipico boton rectangular, tambien puede ser un boton para marcar estados (checkbox), un boton para seleccionar opciones (option button),
                                un control para agrupar otros controles (group box o frame), o un boton del tipo split (algo similar a una combobox estatica pero personalizable), etc...\n

                                La verdad es que si yo hubiera programado dichos controles los hubiera hecho de forma separada, pero eso es otro cantar.\n

                                En definitiva si quieres controlar un CheckBox, o un OptionButton, o un GroupBox, o un SplitButton, o un LinkButton debes utilizar esta clase. Todos los controles que se ven en la siguiente imagen son buttons de algun tipo :
                                <img src='BUTTONS.png' alt='Tipos de Button' />

    \section notificaciones     Notificaciones disponibles :
                                El button manda una serie de notificaciones a traves de los mensajes WM_COMMAND y WM_NOTIFY, que estan todas encapsuladas dentro de la DWL, a continuacin tienes una lista detallada de las notificaciones disponibles, y de que funcin debes utilizar para acceder a ellas :
                                <ul>
                                    <li>
                                        <b>BCN_DROPDOWN</b> Indica que se ha presionado en la parte de la flecha de un boton del tipo split. A partir de esta notificacion podemos por ejemplo crear un menu y desplegarlo.\n
                                        Para obtener esta notificacin en la ventana padre debes sobre-escribir la siguiente funcin : DWL::DEventosPadre::Evento_Button_Desplegar().\n\n
                                    </li>
                                    <li>
                                        <b>BCN_HOTITEMCHANGE</b> Indica que el mouse ha entrado o ha salido del control.\n
                                        Para obtener esta notificacin en la ventana padre debes sobre-escribir las siguientes funcin : \n
                                        Cuando el mouse ha entrado DWL::DEventosPadre::Evento_Button_Mouse_Entrando().\n
                                        Cuando el mouse ha salido DWL::DEventosPadre::Evento_Button_Mouse_Saliendo().\n\n
                                    </li>
                                    <li>
                                        <b>BN_CLICKED</b> Indica que el usuario ha hecho click en el boton.\n
                                        Para obtener esta notificacin en la ventana padre debes sobre-escribir la siguiente funcin : DWL::DEventosPadre::Evento_Button_Mouse_Click().\n\n
                                    </li>
                                    <li>
                                        <b>BN_DBLCLK</b> y <b>BN_DOUBLECLICKED</b> Indica que se ha hecho doble click en el boton.\n
                                        Para obtener esta notificacin en la ventana padre debes sobre-escribir la siguiente funcin : DWL::DEventosPadre::Evento_Button_Mouse_DobleClick().\n
                                        <b>NOTA :</b> Esta notificacion solo estara disponible si el button tiene el estilo BS_NOTIFY activado.\n\n
                                    </li>
                                    <li>
                                        <b>BN_KILLFOCUS</b> Indica que el boton ha perdido el foco.\n
                                        Para obtener esta notificacin en la ventana padre debes sobre-escribir la siguiente funcin : DWL::DEventosPadre::Evento_Button_Foco_Perdido().\n
                                        <b>NOTA :</b> Esta notificacion solo estara disponible si el button tiene el estilo BS_NOTIFY activado.\n\n
                                    </li>
                                    <li>
                                        <b>BN_SETFOCUS</b> Indica que el boton ha recibido el foco.\n
                                        Para obtener esta notificacin en la ventana padre debes sobre-escribir la siguiente funcin : DWL::DEventosPadre::Evento_Button_Foco_Obtenido().\n
                                        <b>NOTA :</b> Esta notificacion solo estara disponible si el button tiene el estilo BS_NOTIFY activado.\n
                                    </li>
                                </ul>
                                Las notificaciones del Button que no esten nombradas aqu las puedes obtener mediante la funcines virtuales DWL::DEventosPadre::Evento_Notificacion() y DWL::DEventosPadre::Evento_Comando().\n

                                Para mas informacin sobre las notificaciones del Button puedes consultar el siguente enlace : <a href='http://msdn.microsoft.com/en-us/library/bb775943(v=vs.85).aspx' target="_blank">MSDN Button</a>.\n

                                Por ultimo hay que remarcar que dispones de un ejemplo donde puedes ver como crear todos los tipos de Button disponibles. Para ver el ejemplo abre el proyecto <i>"DWL 2012 Ejemplos.sln"</i>, y establece como proyecto de inicio el proyecto <i>"Ejemplo Button"</i>.\n\n

    \subpage NotasButton       "Notas de la versin".
                                <hr>
    \author                     devildrey33
    \date                       Creado el [10/01/2012], ultima modificacin el [09/02/2012]
	\remarks	                Archivo creado por devildrey33 para http://www.devildrey33.es														                                \n
				                Este archivo es parte de la DWL (Devildrey33 Windows Library) y se distribuye bajo la licencia GPL, para mas informacin consulta estos enlaces :	\n
				                 - http://www.gnu.org/licenses/gpl.html     (Ingles, documento oficial)																	            \n
				                 - http://www.viti.es/gnu/licenses/gpl.html (Castellano, traduccin no oficial)															            \n
				                 - http://www.softcatala.cat/wiki/GPL3		(Catal, traduccin no oficial)																            \n
*/

#ifndef DBUTTON_H
    #define DBUTTON_H
    
    #include "DBaseControlWin.h"
    #include "DString.h"
    #include "DButton_Estilos.h"

    namespace DWL {
        //! Clase para controlar los distintos tipos de Button de windows.
        class DButton : public DBaseControlWin {
          public:
            ////////////////////////////////////// DObjeto
            /*! Esta funcin se utiliza obtener el nombre de esta clase para fines de depuracin.
                    \brief      Funcin para obtener acceso al nombre de la clase.
                    \fn         virtual const TCHAR *DWL::DButton::Objeto_Nombre(void);
                    \return     Esta funcin devuelve un string terminado con 0.
            */
            /*! Esta funcin se utiliza obtener la ID de esta clase para fines de depuracin.
                    \brief      Funcin para obtener acceso a la ID de la clase.
                    \fn         virtual const DEnum_Objeto DWL::DButton::Objeto_ID(void);
                    \return     Esta funcin devuelve el ID de esta clase.
            */
            DWL_DECLARAR_DOBJETO(TEXT("DButton"), DEnum_Objeto_Button);
            //////////////////////////////////////

                                                //! Constructor
                                                /*! Constructor.
                                                        \fn			DButton(void);
                                                        \return     No devuelve nada
                                                */
                                                DButton(void) : DBaseControlWin() { };

                                                //! Destructor
                                                /*! Destructor.
                                                        \fn		   ~DButton(void);
                                                        \return     No devuelve nada
                                                */
                                               ~DButton(void) { Destruir(); };

                                                //! Funcin para crear el Button (CreateWindowEx).
                                                /*! Esta funcin se usa para crear un Button dinamicamente.
                                                        \fn         HWND Crear(DBaseWnd *nPadre, const TCHAR *nTexto, const int cX, const int cY, const int cAncho, const int cAlto, const UINT cID, DWORD nEstilos);
                                                        \param[in]  nPadre               Clase DBaseWnd que contendra el button.
                                                        \param[in]  nTexto               Texto para el Button.
                                                        \param[in]  cX                   Coordenada X relativa a la ventana padre.
                                                        \param[in]  cY                   Coordenada Y relativa a la ventana padre.
                                                        \param[in]  cAncho               Ancho del Button en pixeles.
                                                        \param[in]  cAlto                Altura del Button en pixeles
                                                        \param[in]  cID                  ID para poder identificar el Button en sus eventos.
                                                        \param[in]  nEstilos             Estilos estandar para el Button.
                                                        \return     Devuelve el HWND del Button o NULL en caso de error.
                                                */
		    HWND	    					    Crear(DBaseWnd *nPadre, const TCHAR *nTexto, const int cX, const int cY, const int cAncho, const int cAlto, const UINT cID, DWORD nEstilos);

                                                //! Funcin para crear el Button (CreateWindowEx).
                                                /*! Esta funcin se usa para crear un Button dinamicamente.
                                                        \fn         HWND Crear(DBaseWnd *Padre, const TCHAR *nTexto, const int cX, const int cY, const int cAncho, const int cAlto, const UINT cID, DButton_Estilos *nEstilos);
                                                        \param[in]  nPadre               Clase DBaseWnd que contendra el button.
                                                        \param[in]  nTexto               Texto para el Button.
                                                        \param[in]  cX                   Coordenada X relativa a la ventana padre.
                                                        \param[in]  cY                   Coordenada Y relativa a la ventana padre.
                                                        \param[in]  cAncho               Ancho del Button en pixeles.
                                                        \param[in]  cAlto                Altura del Button en pixeles
                                                        \param[in]  cID                  ID para poder identificar el Button en sus eventos.
                                                        \param[in]  nEstilos             Puntero a la clase DButton_Estilos que contiene los estilos para el Button.
                                                        \return     Devuelve el HWND del Button o NULL en caso de error.
                                                        \remarks    Esta funcin es mas para depuracin de estilos que para uso cootidiano, ya que las clase DButton_Estilos permite revisar los estilos del button con mucha ms facilidad, pero tambien consumen ms memoria.
                                                */
		    HWND	    					    Crear(DBaseWnd *nPadre, const TCHAR *nTexto, const int cX, const int cY, const int cAncho, const int cAlto, const UINT cID, DButton_Estilos *nEstilos);

                                                //! Funcin para asignar esta clase a un Button de un dialogo.
                                                /*! Esta funcin se usa para asignar esta clase a un Button de un dialogo.
                                                        \fn         HWND Asignar(DBaseWnd *Padre, const UINT cID);
                                                        \param[in]  nPadre      Clase DBaseWnd que contendra el button.
                                                        \param[in]  cID         ID para poder identificar el Button en sus eventos.
                                                        \return     Devuelve el HWND del Button o NULL en caso de error.
                                                        \remarks    Esta funcin solo debe utilizarse si tenemos un Button en un dialogo de los recursos.
                                                */
		    HWND							    Asignar(DBaseWnd *nPadre, const UINT cID);

                                                //! Funcin para asignar el texto del button.
                                                /*! Esta funcin se usa para asignar el texto del button.
                                                        \fn         void AsignarTexto(const TCHAR *nTexto);
                                                        \param[in]  nTexto   Nuevo texto para el button.
                                                        \return     No devuelve nada.
                                                        \sa         ObtenerTexto().
                                                */
            void                                AsignarTexto(const TCHAR *nTexto);

                                                //! Funcin para obtener el texto del button.
                                                /*! Esta funcin se usa para obtener el texto del button.
                                                        \fn         void ObtenerTexto(DString &nTexto);
                                                        \param[out] nTexto   Clase DString donde se almacenara el texto del button.
                                                        \return     No devuelve nada.
                                                        \sa         AsignarTexto().
                                                */
            void                                ObtenerTexto(DString &nTexto);

                                                //! Funcin que devuelve la fuente que se esta utilizando en el Button.
                                                /*! Esta funcin se utiliza para obtener la fuente que se esta utilizando para pintar el texto en el Button.
                                                        \fn         inline HFONT Fuente(void);
                                                        \return     Devuelve la fuente que se esta utilizando en el Button.
                                                        \remarks    Ni se te ocurra borrar esta fuente como mnimo hasta que no sea utilizada por el Button.
                                                */
		    inline HFONT					    Fuente(void) {
												    return (HFONT)SendMessage(_hWnd, WM_GETFONT, 0, 0);
											    };

                                                //! Funcin para asignar la fuente del Button.
                                                /*! Esta funcin se utiliza para asignar la fuente que se utilizara para pintar el texto en el Button.
                                                        \fn         inline void Fuente(HFONT nFuente);
                                                        \param[in]  nFuente     HFONT que pasara a utilizar el Button.
                                                        \return     No devuelve nada.
                                                */
		    inline void						    Fuente(HFONT nFuente) {
												    SendMessage(_hWnd, WM_SETFONT, (WPARAM)nFuente , 0);
											    };

                                                //! Funcin para obtener el estado marcado de un Button.
                                                /*! Esta funcin se utiliza para obtener el estado marcado de un Button (valida tambien para el CheckBox y el OptionButton).
                                                        \fn         inline const BOOL Marcado(void);
                                                        \return     Devuelve BST_UNCHECKED si no esta marcado, BST_CHECKED si esta marcado, y BST_INDETERMINATE si esta completamente pintado.
                                                        \remarks    Solo puede devolver BST_INDETERMINATE si el button en cuestion tiene el estilo BS_AUTO3STATE o BS_3STATE.
                                                */
            inline const BOOL                   Marcado(void) {
                                                    return static_cast<BOOL>(SendMessage(_hWnd, BM_GETCHECK, 0, 0));
                                                };

                                                //! Funcin para asignar el estado marcado de un Button.
                                                /*! Esta funcin se utiliza para asignar el estado marcado de un Button (valida tambien para el CheckBox y el OptionButton).
                                                        \fn         inline void Marcar(const BOOL nMarcar);
                                                        \param[in]  nMarcar     BST_UNCHECKED para desmarcar el boton, BST_CHECKED para marcar el boton, y BST_INDETERMINATE para pintarlo completamente.
                                                        \return     No devuelve nada.
                                                        \remarks    Solo se puede asignar BST_INDETERMINATE si el button en cuestion tiene el estilo BS_AUTO3STATE o BS_3STATE.
                                                */
            inline void                         Marcar(const BOOL nMarcar) {
                                                    SendMessage(_hWnd, BM_SETCHECK, nMarcar, 0);
                                                };

                                                //! Funcin para mostrar el icono de elevacin de privilegios en el button.
                                                /*! Esta funcin se utiliza para mostrar el icono de elevacin de privilegios en el button.
                                                        \fn         inline void IconoElevarPrivilegios(const BOOL nIcono);
                                                        \param[in]  nIcono     TRUE para mostrar la imagen, FALSE para no mostrarla.
                                                        \return     No devuelve nada.
                                                        \remarks    Esta funcin solo muestra el icono de elvacion de privilegios en el boton, si luego el boton no se utiliza para ese fin es cosa del programador.
                                                */
            inline void                         IconoElevarPrivilegios(const BOOL nIcono) {
                                                    SendMessage(_hWnd, BCM_SETSHIELD, 0, nIcono);
                                                };

                                                //! Funcin que retorna los estilos del Button.
                                                /*! Esta funcin introduce en el parametro nEstilos los estilos que contiene el Button.
                                                        \fn         void ObtenerEstilos(DButton_Estilos &nEstilos);
                                                        \param[out] nEstilos   Clase DButton_Estilos en la que se depositaran los estilos del Button.
                                                        \return     Esta funcin no devuelve nada.
                                                */
            void                                ObtenerEstilos(DButton_Estilos &nEstilos);

                                                //! Funcin que retorna los estilos del Button.
                                                /*! Esta funcin retorna todos los estilos del Button.
                                                        \fn         DWORD ObtenerEstilos(void);
                                                        \return     Devuelve los estilos del Button.
                                                */
            DWORD                               ObtenerEstilos(void);

                                                //! Funcin para asignar los estilos del Button.
                                                /*! Esta funcin permite asignar uno o mas estilos mediante el parametro nEstilos previamente preparado
                                                        \fn         void AsignarEstilos(DButton_Estilos &nEstilos)
                                                        \param[in]  nEstilos   Clase DButton_Estilos que contiene los nuevos estilos a asignar.
                                                        \return     No devuelve nada.
                                                        \remarks    Esta funcin esta pensada para modificar todos los estilos de golpe, por lo que si quieres modificar un solo estilo conservando los valores de los demas,
                                                                    primero deberas obtener los estilos mediante la funcin ObtenerEstilos(), para luego modificar lo que te interese, y finalmente pasar la
                                                                    clase DButton_Estilos a esta funcin.
                                                */
            void                                AsignarEstilos(DButton_Estilos &nEstilos);

                                                //! Funcin para asignar los estilos del Button.
                                                /*! Esta funcin permite asignar uno o mas estilos mediante el parametro nEstilos previamente preparado.
                                                        \fn         void AsignarEstilos(DWORD nEstilos)
                                                        \param[in]  nEstilos   DWORD que contiene los nuevos estilos a asignar.
                                                        \return     No devuelve nada.
                                                        \remarks    Esta funcin esta pensada para modificar todos los estilos de golpe, por lo que si quieres modificar un solo estilo conservando los valores de los demas,
                                                                    primero deberas obtener los estilos mediante la funcin ObtenerEstilos(), para luego modificar lo que te interese, y finalmente pasar los estilos a esta funcin
                                                */
            void                                AsignarEstilos(DWORD nEstilos);

                                                //! Funcin que devuelve que tipo de clase BaseWnd es esta clase
                                                /*! Esta funcin devuelve que tipo de clase BaseWnd es esta clase
                                                        \fn         virtual const DEnum_TipoBaseWnd TipoBaseWnd(void);
                                                        \returns    Devuelve un valor de DEnum_TipoBaseWnd
                                                        \remars     Esta funcin identifica que tipo de ventana o control se esta utilizando.
                                                */
            virtual const DEnum_TipoBaseWnd     TipoBaseWnd(void)   { return DEnum_TipoBaseWnd_Button; };

        };                                      //
        ////////////////////////////////////////// FIN Clase DButton
    };

#endif

/*! \page   NotasButton     Notas de las versiones de Button
<b>Versin 1.01 [14/02/2012]</b>
<ul>
    <li>Se ha re-emplazado el parametro "HWND hWndPadre" de las funciones "Crear" y "Asignar" por DBaseWnd *nPadre.</li>
</ul>

<b>Versin 1.0 [09/02/2012]</b>
<ul>
    <li>Creada clase para los estilos del Button DButton_Estilos.</li>
    <li>Creados 7 eventos para el boton.</li>
    <li>Agregada funcin IconoElevarPrivilegios que permite mostrar el icono de elevacin de privilegios.</li>
    <li>Documentado todo el archivo DBoton.h al estilo doxygen.</li>
</ul>
*/